# PCI Header script
#
# Copyright 2006, Haiku.
# Distributed under the terms of the MIT License.
#
# Authors:
#		John Drinkwater, john@nextraweb.com
#
# Use with http://pciids.sourceforge.net/pci.ids
# run as: awk -v HEADERFILE=pcihdr.h -f pci-header.awk pci.ids

BEGIN {

	# field separator, defining because user could have overridden
	FS = " "

	# Pass this in from outside with -v HEADERFILE=filenametouse
	# we require pcihdr.h for our system
	ofile = HEADERFILE

	# possibly use this in the future
	cleanvalues = "[^A-Za-z0-9{}\"'&@?!*.,:;+<> \\t\\/_\\[\\]=#()-]"
	# ToDo: currently IDs aren't checked, we dumbly assume the source is clean

	# descriptive output header
	print "#if 0" > ofile
	print "#\tPCIHDR.H: PCI Vendors, Devices, and Class Type information\n#" > ofile
	print "#\tGenerated by pci-header.awk, source data from the following URI:\n#\thttp://pciids.sourceforge.net/pci.ids\n#" > ofile
	print "#\tHeader created on " strftime( "%A, %d %b %Y %H:%M:%S %Z", systime() ) > ofile
	print "#endif" > ofile

	# and we start with vendors..
	print "\ntypedef struct _PCI_VENTABLE\n{\n\tunsigned short\tVenId ;\n\tconst char *\tVenFull ;\n\tconst char *\tVenShort ;\n}  PCI_VENTABLE, *PPCI_VENTABLE ;\n" > ofile
	print "PCI_VENTABLE\tPciVenTable [] =\n{" > ofile
}

# matches vendor - starts with an id as first thing on the line
# because this occurs first in the header file, we output it without worry
/^[[:xdigit:]][[:xdigit:]][[:xdigit:]][[:xdigit:]] / { 

	if ( vendorcount++ > 0 ) { 
		formatting = ",\n"
	} else {
		formatting = ""
	}

	# store vendor ID for possible devices afterwards
	vendorid = $1
	vendor = substr($0, 7)
	gsub( /\"/, "&&", vendor )

	printf formatting "\t{ 0x" vendorid ", \"" vendor "\", \"\" }" > ofile
}

# matches device 
/^\t[[:xdigit:]][[:xdigit:]][[:xdigit:]][[:xdigit:]] / { 

	device = substr($0, 8)
	gsub( /\\/, "&&", device )
	gsub( /\"/, "&&", device )

	# store device ID for possible devices afterwards
	deviceid = $1
	devicecount++
	devices[devicecount, 1] = vendorid
	devices[devicecount, 2] = $1
	devices[devicecount, 3] = 0
	devices[devicecount, 4] = 0
	devices[devicecount, 5] = device 
}

# matches subvendor device
/^\t\t[[:xdigit:]][[:xdigit:]][[:xdigit:]][[:xdigit:]] / { 

	device = substr($0, 14)
	gsub( /\"/, "\\\"", device )

	devicecount++
	devices[devicecount, 1] = vendorid
	devices[devicecount, 2] = deviceid
	devices[devicecount, 3] = $1 
	devices[devicecount, 4] = $2
	devices[devicecount, 5] = device 
}

# match device class - store data for later
/^C [[:xdigit:]][[:xdigit:]]  / { 

	class = $2
	classname = substr($0, 7)
	gsub( /\"/, "\\\"", classname )
}

# match subclass, use device class data captured earlier, and output
/^\t[[:xdigit:]][[:xdigit:]]  / {

	subclass = $1
	subclassname = substr($0, 6)
	gsub( /\"/, "\\\"", subclassname )

	classcount++
	classes[classcount, 1] = class
	classes[classcount, 2] = subclass
	classes[classcount, 3] = "00"
	classes[classcount, 4] = classname
	classes[classcount, 5] = subclassname
	classes[classcount, 6] = ""
} 

# match programming interface
/^\t\t[[:xdigit:]][[:xdigit:]]  / {

	proginterface = $1
	proginterfacename = substr($0, 7)
	gsub( /\"/, "\\\"", proginterfacename )

	classcount++
	classes[classcount, 1] = class
	classes[classcount, 2] = subclass
	classes[classcount, 3] = proginterface
	classes[classcount, 4] = classname
	classes[classcount, 5] = subclassname
	classes[classcount, 6] = proginterfacename
} 

# We've processed the file, now output.
END {

	print "\n};\n\n// Use this value for loop control during searching:\n#define\tPCI_VENTABLE_LEN\t(sizeof(PciVenTable)/sizeof(PCI_VENTABLE))\n" > ofile

	if ( devicecount > 0 ) {

		print "typedef struct _PCI_DEVTABLE\n{\n\tunsigned short	VenId ;\n\tunsigned short	DevId ;\n\tunsigned short\tSubVenId ;\n\tunsigned short\tSubDevId ;\n\tconst char *\tChipDesc ;\n\tconst char *\tChip;\n}  PCI_DEVTABLE, *PPCI_DEVTABLE ;\n"  > ofile
		print "PCI_DEVTABLE\tPciDevTable [] =\n{" > ofile
		for (i = 1; i <= devicecount; i++) {

			if (i != 1) {
				formatting = ",\n"
			} else {
				formatting = ""
			}
			printf formatting "\t{ 0x" devices[i, 1] ", 0x" devices[i, 2] ", 0x" devices[i, 3] ", 0x" devices[i, 4] ", \"" devices[i, 5] "\" }" > ofile
		}
		print "\n} ;\n\n// Use this value for loop control during searching:\n#define	PCI_DEVTABLE_LEN	(sizeof(PciDevTable)/sizeof(PCI_DEVTABLE))\n" > ofile

	}
	
	if ( classcount > 0 ) {
		print "typedef struct _PCI_CLASSCODETABLE\n{\n\tunsigned char	BaseClass ;\n\tunsigned char	SubClass ;\n\tunsigned char	ProgIf ;" > ofile
		print "\tconst char *\t\tBaseDesc ;\n\tconst char *\t\tSubDesc ;\n\tconst char *\t\tProgDesc ;\n}  PCI_CLASSCODETABLE, *PPCI_CLASSCODETABLE ;\n" > ofile
		print "PCI_CLASSCODETABLE PciClassCodeTable [] =\n{" > ofile
		currentclass = classes[1, 1]
		for (i = 1; i <= classcount; i++) {
			
			if (i != 1) { 
				formatting = ",\n"
			} else { 
				formatting = ""
			}

			# pretty print separate classes
			if ( currentclass != classes[i, 1] ) { 
				formatting = formatting "\n"
				currentclass = classes[i, 1]
			}
			
			# if the next item has the same details, we know we're to skip ourselves 
			# this is because the programming interface name needs to be used, and we dont have it ourselves
			if ( ( classes[i, 1] != classes[i+1, 1] ) || ( classes[i, 2] != classes[i+1, 2] ) || ( classes[i, 3] != classes[i+1, 3] ) ) {
				printf formatting "\t{ 0x" classes[i, 1] ", 0x" classes[i, 2] ", 0x" classes[i, 3] ", \"" classes[i, 4] "\", \"" classes[i, 5]  "\", \"" classes[i, 6] "\" }" > ofile
			}
		}
		print "\n} ;\n\n// Use this value for loop control during searching:\n#define	PCI_CLASSCODETABLE_LEN	(sizeof(PciClassCodeTable)/sizeof(PCI_CLASSCODETABLE))\n" > ofile

	}

	# this is rather ugly, maybe we should include this in a seperate file, and pull it in ?
	print "const char *\tPciCommandFlags [] =\n{\n\t\"I/O Access\",\n\t\"Memory Access\",\n\t\"Bus Mastering\",\n\t\"Special Cycles\",\n\t\"Memory Write & Invalidate\",\n\t\"Palette Snoop\",\n\t\"Parity Errors\",\n\t\"Wait Cycles\",\n\t\"System Errors\",\n\t\"Fast Back-To-Back\",\n\t\"Reserved 10\",\n\t\"Reserved 11\",\n\t\"Reserved 12\",\n\t\"Reserved 13\",\n\t\"Reserved 14\",\n\t\"Reserved 15\"\n} ;\n" > ofile
	print "// Use this value for loop control during searching:\n#define	PCI_COMMANDFLAGS_LEN	(sizeof(PciCommandFlags)/sizeof(char *))\n" > ofile
	print "const char *\tPciStatusFlags [] =\n{\n\t\"Reserved 0\",\n\t\"Reserved 1\",\n\t\"Reserved 2\",\n\t\"Reserved 3\",\n\t\"Reserved 4\",\n\t\"66 MHz Capable\",\n\t\"User-Defined Features\",\n\t\"Fast Back-To-Back\",\n\t\"Data Parity Reported\",\n\t\"\",\n\t\"\",\n\t\"Signalled Target Abort\",\n\t\"Received Target Abort\",\n\t\"Received Master Abort\",\n\t\"Signalled System Error\",\n\t\"Detected Parity Error\"\n} ;\n" > ofile
	print "// Use this value for loop control during searching:\n#define	PCI_STATUSFLAGS_LEN	(sizeof(PciStatusFlags)/sizeof(char *))\n" > ofile
	print "const char *\tPciDevSelFlags [] =\n{\n\t\"Fast Devsel Speed\",\n\t\"Medium Devsel Speed\",\n\t\"Slow Devsel Speed\",\n\t\"Reserved 9&10\"\n} ;\n" > ofile
	print "// Use this value for loop control during searching:\n#define	PCI_DEVSELFLAGS_LEN	(sizeof(PciDevSelFlags)/sizeof(char *))\n\n" > ofile

	close(ofile)
}


